#![allow(nonstandard_style)]

extern crate plotters;
use plotters::prelude::*;
use plotters::style::FontStyle;
use std::collections::HashMap;
use std::{fs, path, vec};
use rand::Rng;
use std::time::Instant;


struct Node {
    id: u64,
    neighbors: Vec<u64>,
}

// Extracting data
fn readFileString(file_path : String) -> String{
    let content = fs::read_to_string(file_path);
    
    
    match content {
        Ok(data) => {
            String::from(data)
        },
        Err(_) => String::from("not ok"),
    }
}

fn extractNodeFromData(dataBlob : String) -> Node{
    let Split: Vec<&str> = dataBlob.split(":").collect();
    let id: u64 = Split[0].parse::<u64>().unwrap();

    let mut neigbors: Vec<u64> = Vec::new();
    let neigborsStrings : Vec<&str> = Split[1].split(",").collect();

    for neib in neigborsStrings{
        neigbors.push(neib.parse::<u64>().unwrap());
    }

    let Node : Node = Node { id: id, neighbors: neigbors};

    return Node;
}

fn extractPathFromData(dataBlob : String) -> Vec<u64>{
    let mut neigbors: Vec<u64> = Vec::new();
    let neigborsStrings : Vec<&str> = dataBlob.split(",").collect();

    for neib in neigborsStrings{
        neigbors.push(neib.parse::<u64>().unwrap());
    }

    return neigbors;
}

fn extractListData(file_path : String) -> (Vec<Node>, Vec<u64>) {
    let StringData = readFileString(file_path);

    let lines: Vec<&str> = StringData.split('\n').collect();

    let mut NodesList: Vec<Node> = Vec::new();
    let mut PathList: Vec<u64> = Vec::new();

    for line in lines {
        if line.contains(":") {
            NodesList.push(extractNodeFromData(String::from(line)))
        } else if line.contains(",") {
            PathList = extractPathFromData(String::from(line));
        }
    }

    return (NodesList, PathList);
}

// Utilities
fn find_node_by_id(node_list: &Vec<Node>, id: u64) -> Option<&Node> {
    node_list.iter().find(|&node| node.id == id)
}

fn isNotInList(tryingToFind : u64, listOfNeighb : &Vec<u64>) -> bool {
    if listOfNeighb.contains(&tryingToFind) {
        return false;
    }
    return true;
}

// Main algorithm for finding if its true
fn checkIfLoopIsCorrect(nodeList : &Vec<Node>, pathList : &Vec<u64>) -> bool {
    if pathList.last() != Some(&pathList[0]) { return false; }

    let mut alreadyUsedNodeList : Vec<u64> = Vec::new();

    for i in 0..(pathList.len()-1) {
        let nextNodeId = pathList[i+1];
        
        let currentNode = match find_node_by_id(&nodeList,pathList[i]) {
            Some(node_ref) => node_ref, // currentNode is now &Node
            None => { 
                println!("There is no node with id that is found in path\n");
                return false; 
            }
        };

        if alreadyUsedNodeList.contains(&currentNode.id) {
            println!("Point was used more then 1 time\n");
            return false;
        }

        if isNotInList(nextNodeId, &currentNode.neighbors) {
            println!("path list is not a complete / viable line to make a cicle\n");
            return false;
        }

        alreadyUsedNodeList.push(currentNode.id);
    }

    return true;
}

fn _TestingPrint(path_list: &Vec<u64>) {
    let mut print_string = String::from("Path: ");

    for value in path_list {
        print_string.push_str(&value.to_string());
        print_string.push(' '); // optional spacing
    }

    println!("{}", print_string);
}



fn getAllValidLoops(nodeList : &Vec<Node>, mainGrandine : &Vec<u64>) -> Vec<Vec<u64>>{
    let mut results: Vec<Vec<u64>> = Vec::new();
    
    if mainGrandine.len() < 3 {
        return results;
    }


    for i in 0..(mainGrandine.len() - 2) {
        let current_node_id = mainGrandine[i];

        let currentNodeFullGrandine = mainGrandine[i..].to_vec();

        for u in 2..currentNodeFullGrandine.len() {
            let mut loopToCheck = currentNodeFullGrandine[..=u].to_vec(); // Tipo teoriskai galeciau sita padaryt ir be currentNodeFullGrandine
            loopToCheck.push(current_node_id);

            print!("Testing ");
            _TestingPrint(&loopToCheck);

            let isSubLoopViable = checkIfLoopIsCorrect(nodeList, &loopToCheck);
            if isSubLoopViable {
                
                println!("Is A Good Loop \n");
                
                results.push(loopToCheck);
            }
        }
    }

    return results;
}

// DRAWING OUTPUT

fn get_color(max_index: u32, current_index: u32) -> RGBColor {
    let max_color = RGBColor(217, 119, 87);
    // let min_color = RGBColor(89, 79, 66);
    let min_color = RGBColor(217, 119, 87);

    // Avoid division by zero
    if max_index == 0 {
        return min_color;
    }

    // Interpolation factor 0.0 .. 1.0
    let t = current_index as f64 / max_index as f64;

    // Interpolate each channel
    let r = min_color.0 as f64 + t * (max_color.0 as f64 - min_color.0 as f64);
    let g = min_color.1 as f64 + t * (max_color.1 as f64 - min_color.1 as f64);
    let b = min_color.2 as f64 + t * (max_color.2 as f64 - min_color.2 as f64);

    RGBColor(r.round() as u8, g.round() as u8, b.round() as u8)
}

// fn get_color(max_index: u32, current_index: u32) -> RGBColor {
//     let mut rng = rand::rng();

//     let r = rng.random_range(0..=255);
//     let g = rng.random_range(0..=255);
//     let b = rng.random_range(0..=255);

//     RGBColor(r, g, b)
// }

fn draw_image(node_list: Vec<Node>, ListOfPaths : Vec<Vec<u64>>, text_to_output: String) -> Result<(), Box<dyn std::error::Error>> {
    let node_count = node_list.len();
    let mut node_positions: HashMap<u64, (f64, f64)> = HashMap::new();
    

    let radius = 300.0;
    let center_x = 400.0;
    let center_y = 400.0;
    
    for (i, node) in node_list.iter().enumerate() {
        let angle = 2.0 * std::f64::consts::PI * (i as f64) / (node_count as f64);
        let x = center_x + radius * angle.cos();
        let y = center_y + radius * angle.sin();
        node_positions.insert(node.id, (x, y));
    }
    

    let line_height: u32 = 40;


    let root = BitMapBackend::new("graph.png", (800, 900 + (line_height * text_to_output.lines().count() as u32))).into_drawing_area();
    root.fill(&RGBColor(38, 38, 36))?;
    
    let (graph_area, text_area) = root.split_vertically(800);
    
    let mut chart = ChartBuilder::on(&graph_area)
        .margin(10)
        .build_cartesian_2d(0f64..800f64, 0f64..800f64)?;
    
    //Draw inactive
    for node in &node_list {
        if let Some(&(x1, y1)) = node_positions.get(&node.id) {
            for &neighbor_id in &node.neighbors {
                if let Some(&(x2, y2)) = node_positions.get(&neighbor_id) {
                    chart.draw_series(LineSeries::new(
                        vec![(x1, y1), (x2, y2)],
                        ShapeStyle::from(&RGBColor(72, 72, 68)).stroke_width(4),
                    ))?;
                }
            }
        }
    }
    for node in &node_list {
        if let Some(&(x, y)) = node_positions.get(&node.id) {
            // let is_in_path = path_list.contains(&node.id);
            let color = &RGBColor(72, 72, 68) ;
            
            chart.draw_series(PointSeries::of_element(
                vec![(x, y)],
                30,
                color,
                &|coord, size, style| {
                    EmptyElement::at(coord)
                        + Circle::new((0, 0), size, style.filled())
                        + Text::new(
                            format!("{}", node.id),
                            (-10, -10),
                            ("sans-serif", 40).into_font().style(FontStyle::Bold).color(&RGBColor(250, 249, 245)),
                        )
                },
            ))?;
        }
    }

    // Draw Active path edges
    for i in 0..ListOfPaths.len() {
        let path_list: &Vec<u64> = &ListOfPaths[i];

        for i in 0..path_list.len().saturating_sub(1) {
            let node1 = path_list[i];
            let node2 = path_list[i + 1];
            
            if let (Some(&(x1, y1)), Some(&(x2, y2))) = 
                (node_positions.get(&node1), node_positions.get(&node2)) {
                chart.draw_series(LineSeries::new(
                    vec![(x1, y1), (x2, y2)],
                    ShapeStyle::from(&get_color((ListOfPaths.len()-1) as u32, i as u32)).stroke_width(5),
                ))?;
            }
        }
        
        for node in &node_list {
            if let Some(&(x, y)) = node_positions.get(&node.id) {
                let is_in_path = path_list.contains(&node.id);
                let color =  &get_color((ListOfPaths.len()-1) as u32, i as u32);
                
                if is_in_path {
                    chart.draw_series(PointSeries::of_element(
                        vec![(x, y)],
                        30,
                        color,
                        &|coord, size, style| {
                            EmptyElement::at(coord)
                                + Circle::new((0, 0), size, style.filled())
                                + Text::new(
                                    format!("{}", node.id),
                                    (-10, -10),
                                    ("sans-serif", 40).into_font().style(FontStyle::Bold).color(&RGBColor(250, 249, 245)),
                                )
                        },
                    ))?;
                }
            }
        }
    }


    
    
    let start_x :u32= 20;
    let mut start_y:u32 = 30;
    
    
    // Split the text into lines
    for line in text_to_output.lines() {
        text_area.draw_text(
            line,
            &("sans-serif", 30).into_font().color(&RGBColor(250, 249, 245)),
            (start_x as i32, start_y as i32),
        )?;
    
        start_y += line_height; // move down for next line
    }
    
    root.present()?;
    println!("Graph saved to graph.png");
    Ok(())
}

fn generate_text_to_output(path_list: &Vec<u64>, is_loop_viable: &bool) -> String {
    let mut current_string = String::from(" • Loop (");
    let mut index : usize = 0;

    for id in path_list {
        current_string += &(id.to_string());
        if index != (path_list.len()-1) { current_string += ", "}
        index += 1;
    }
    
    current_string += ") is ";
    
    if *is_loop_viable {
        current_string += "a cycle\n";
    } else {
        current_string += "not a cycle\n";
    }
    
    current_string
}

fn generate_full_list_to_output(ListOfViableLoops : &Vec<Vec<u64>>) -> String{
    let mut fullString: String = String::from("List of loops:\n");

    for i in 0..ListOfViableLoops.len() {
        fullString.push_str(&generate_text_to_output(&ListOfViableLoops[i],&true));
    }

    return fullString;
}

fn main() {
    println!("Starting...");
    
    let (NodeList, MainGrandine) = extractListData("./Data.txt".to_string());
    
    let now = Instant::now();
    let ListOfViableLoops = getAllValidLoops(&NodeList,&MainGrandine);
    let elapsed_time = now.elapsed();
    
    let text = generate_full_list_to_output(&ListOfViableLoops);
    println!("{}", text);
    
    
    draw_image(NodeList, ListOfViableLoops, text).expect("Failed to draw graph");

    println!("Job done...");
    println!("Algorithm took {:?}.", elapsed_time);
}